const {
  Plugin,
  PluginSettingTab,
  Setting,
  MarkdownRenderer,
  Modal,
  Notice,
} = require("obsidian");

module.exports = class AskAiPlugin extends Plugin {
  async onload() {
    console.log("Ask AI 插件已加载");

    await this.loadSettings();
    this.addSettingTab(new AskAiSettingTab(this.app, this));

    this.addCommand({
      id: "ask-ai",
      name: "Ask AI about selection",
      editorCallback: (editor) => {
        this.handleSelection(editor);
      },
    });

    this.registerEvent(
      this.app.workspace.on("active-leaf-change", (leaf) => {
        if (leaf?.view?.editor) {
          const cm =
            leaf.view.editor.cm?.dom || leaf.view.editor.cm?.display?.wrapper;
          if (cm) {
            cm.addEventListener("mouseup", (evt) => {
              const selection = leaf.view.editor.getSelection();
              if (selection && selection.trim()) {
                this.showFloatingButton(evt, selection, leaf.view.editor);
              }
            });
          }
        }
      })
    );
  }

  onunload() {
    console.log("Ask AI 插件已卸载");
  }

  async loadSettings() {
    this.settings = Object.assign(
      {
        apis: [],
        defaultApiIndex: 0,
        promptTemplate: `请根据selection，按如下要求执行：selection:\n{{selection}}\n\n要求：\n- 如果\"selection\"是英文单词则**仅翻译并给出其词源, 忽略其他要求**\n- 如果\"selection\"是一段英文，请分别给出直译和接地气的中文翻译,并解释其中关键词句；\n- 如果\"selection\"是中文段落，请通俗易懂地解释其含义；\n- 如有上下文，请结合上下文进行说明, \n**翻译仅限selection内容**。\n\n上下文：{{context}}`,
        defaultFontFamily: ""
      },
      await this.loadData()
    );
  }

  async saveSettings() {
    await this.saveData(this.settings);
  }

  showFloatingButton(evt, selection, editor) {
  const button = document.createElement("button");
  button.textContent = "🤖";
  button.className = "ask-ai-btn";
  button.style.position = "absolute";
  button.style.left = `${evt.pageX + 10}px`;
  button.style.top = `${evt.pageY}px`;
  button.style.zIndex = "9999";
  button.style.width = "25px";
  button.style.height = "25px";
  button.style.fontSize = "1em";
  button.style.borderRadius = "12.5px";

    document.body.appendChild(button);

    const remove = () => button.remove();
    button.onclick = () => {
  if (this.settings.boldOnClick && editor) {
    const selection = editor.getSelection();
    if (selection && !selection.startsWith("**") && !selection.endsWith("**")) {
      editor.replaceSelection(`**${selection}**`);
    }
  }
  new AskModal(this.app, this, selection, editor).open();
  remove();
};


    // 右键：弹出 API 列表
    button.oncontextmenu = (e) => {
      e.preventDefault();

      const menu = document.createElement("div");
      menu.className = "ask-ai-menu";
      menu.style.position = "absolute";
      menu.style.left = `${e.pageX}px`;
      menu.style.top = `${e.pageY}px`;
      menu.style.background = "var(--background-primary)";
      menu.style.border = "1px solid var(--background-modifier-border)";
      menu.style.padding = "4px";
      menu.style.borderRadius = "6px";
      menu.style.zIndex = "10000";

      // 下划线选项
      const underlineItem = document.createElement("div");
      underlineItem.textContent = "下划线";
      underlineItem.style.padding = "4px 8px";
      underlineItem.style.cursor = "pointer";
      underlineItem.style.borderBottom = "1px solid #eee";
      underlineItem.onmouseenter = () => underlineItem.style.background = "var(--background-modifier-hover)";
      underlineItem.onmouseleave = () => underlineItem.style.background = "transparent";
      underlineItem.onclick = () => {
        if (editor) {
          const sel = editor.getSelection();
          if (sel && !sel.startsWith("<u>") && !sel.endsWith("</u>")) {
            editor.replaceSelection(`<u>${sel}</u>`);
          }
        }
        menu.remove();
        remove();
      };
      menu.appendChild(underlineItem);

      // AI API 列表
      this.settings.apis.forEach((api, index) => {
        const item = document.createElement("div");
        item.textContent = api.name || `API ${index + 1}`;
        item.style.padding = "4px 8px";
        item.style.cursor = "pointer";
        item.onmouseenter = () =>
          (item.style.background = "var(--background-modifier-hover)");
        item.onmouseleave = () =>
          (item.style.background = "transparent");

        item.onclick = () => {
          new AskModal(this.app, this, selection, editor, index).open();
          menu.remove();
          remove();
        };

        menu.appendChild(item);
      });

      document.body.appendChild(menu);

      // 点击外部时移除菜单
      const cleanup = () => {
        menu.remove();
        document.removeEventListener("click", cleanup);
      };
      document.addEventListener("click", cleanup);
    };

    setTimeout(() => {
      document.addEventListener("click", remove, { once: true });
    }, 0);
  }

  async handleSelection(editor) {
    const selectedText = editor.getSelection();
    if (!selectedText) {
      new Notice("没有选中文字");
      return;
    }
    new AskModal(this.app, this, selectedText, editor).open();
  }
};

class AskModal extends Modal {
  constructor(app, plugin, query, editor, apiIndex = null) {
    super(app);
    this.plugin = plugin;
    this.query = query;   // 选中的内容
    this.editor = editor; // 编辑器对象
    this.apiIndex = apiIndex;
    this.abortController = null;
    this.messages = [];   // 对话历史
    this._drag = { active: false, offsetX: 0, offsetY: 0 };
  }

  async onOpen() {
    const { contentEl, modalEl } = this;
    contentEl.empty();
    contentEl.addClass("ask-ai-modal");
    contentEl.style.userSelect = "text";

    // 让弹窗可拖动
    modalEl.style.position = "fixed";
    modalEl.style.zIndex = "99999";
    modalEl.style.left = "50%";
    modalEl.style.top = "20%";
    modalEl.style.transform = "translate(-50%, 0)";

    // 拖动逻辑
    let dragStartX, dragStartY, dragStartLeft, dragStartTop;
    const onDragMouseDown = (e) => {
      if (!e.target.classList.contains("ask-ai-modal-header")) return;
      this._drag.active = true;
      dragStartX = e.clientX;
      dragStartY = e.clientY;
      const rect = modalEl.getBoundingClientRect();
      dragStartLeft = rect.left;
      dragStartTop = rect.top;
      document.addEventListener("mousemove", onDragMouseMove);
      document.addEventListener("mouseup", onDragMouseUp);
      e.preventDefault();
    };
    const onDragMouseMove = (e) => {
      if (!this._drag.active) return;
      const dx = e.clientX - dragStartX;
      const dy = e.clientY - dragStartY;
      modalEl.style.left = `${dragStartLeft + dx}px`;
      modalEl.style.top = `${dragStartTop + dy}px`;
      modalEl.style.transform = "none";
    };
    const onDragMouseUp = () => {
      this._drag.active = false;
      document.removeEventListener("mousemove", onDragMouseMove);
      document.removeEventListener("mouseup", onDragMouseUp);
    };
    modalEl.addEventListener("mousedown", onDragMouseDown);

    // 标题
    const header = contentEl.createEl("h3", { text: "AI 回答中..." });
    header.classList.add("ask-ai-modal-header");
    header.style.cursor = "move";

    // 输出区
  this.textarea = contentEl.createEl("textarea");
  this.textarea.addClass("ask-ai-textarea");
    this.textarea.style.width = "100%";
    this.textarea.style.height = "300px";
    this.textarea.style.whiteSpace = "pre-wrap";
    this.textarea.style.lineHeight = "1.6";
    this.textarea.style.fontFamily = this.plugin.settings.defaultFontFamily || "";

    // 清除格式、复制、字体大小按钮
  const formatBtnRow = contentEl.createDiv();
  formatBtnRow.addClass("ask-ai-format-row");
    formatBtnRow.style.display = "flex";
    formatBtnRow.style.justifyContent = "flex-end";
    formatBtnRow.style.alignItems = "center";
    formatBtnRow.style.marginBottom = "10px";

    // 字体大小按钮
    const fontDecBtn = document.createElement("button");
    fontDecBtn.textContent = "A-";
    fontDecBtn.style.padding = "4px 8px";
    fontDecBtn.style.fontSize = "0.8em";
    fontDecBtn.style.marginRight = "8px";
    formatBtnRow.appendChild(fontDecBtn);

    const fontIncBtn = document.createElement("button");
    fontIncBtn.textContent = "A+";
    fontIncBtn.style.padding = "4px 8px";
    fontIncBtn.style.fontSize = "0.8em";
    fontIncBtn.style.marginRight = "8px";
    formatBtnRow.appendChild(fontIncBtn);

    // 插入 >[!]按钮
    const insertBtn = document.createElement("button");
    insertBtn.textContent = ">[!]";
    insertBtn.style.padding = "4px 8px";
    insertBtn.style.fontSize = "0.8em";
    insertBtn.style.marginRight = "8px";
    insertBtn.onclick = () => {
      const selectedText = this.query || "";
      this.textarea.value = `>[!${selectedText}]\n` + this.textarea.value;
    };
    formatBtnRow.appendChild(insertBtn);

    // 清除格式按钮
    const clearFormatBtn = document.createElement("button");
    clearFormatBtn.textContent = "清除格式";
    clearFormatBtn.style.padding = "4px 8px";
    clearFormatBtn.style.fontSize = "0.8em";
    clearFormatBtn.style.marginRight = "8px";
    clearFormatBtn.onclick = () => {
      const currentText = this.textarea.value;
      const cleanedText = currentText
        .replace(/\*\*(.*?)\*\*/g, '$1')
        .replace(/\*(.*?)\*/g, '$1')
        .replace(/~~(.*?)~~/g, '$1')
        .replace(/`(.*?)`/g, '$1')
        .replace(/^#+\s+/gm, '')
        .replace(/^-{3,}$/gm, '')
        .replace(/\n{2,}/g, '\n');
      this.textarea.value = cleanedText + "\n";
    };
    formatBtnRow.appendChild(clearFormatBtn);

    // 复制按钮
    const copyBtn = document.createElement("button");
    copyBtn.textContent = "复制";
    copyBtn.style.padding = "4px 8px";
    copyBtn.style.fontSize = "0.8em";
    copyBtn.style.marginRight = "8px";
    copyBtn.onclick = async () => {
      let textToCopy = "";
      const textarea = this.textarea;
      if (textarea.selectionStart !== textarea.selectionEnd) {
        textToCopy = textarea.value.substring(textarea.selectionStart, textarea.selectionEnd);
      } else {
        textToCopy = textarea.value;
      }
      try {
        await navigator.clipboard.writeText(textToCopy);
        copyBtn.textContent = "已复制";
        setTimeout(() => { copyBtn.textContent = "复制"; }, 1200);
      } catch (err) {
        copyBtn.textContent = "失败";
        setTimeout(() => { copyBtn.textContent = "复制"; }, 1200);
      }
    };
    formatBtnRow.appendChild(copyBtn);

    // 字体大小调整逻辑
    let fontSize = 14;
    this.textarea.style.fontSize = fontSize + "px";
    fontIncBtn.onclick = () => {
      fontSize = Math.min(fontSize + 2, 32);
      this.textarea.style.fontSize = fontSize + "px";
    };
    fontDecBtn.onclick = () => {
      fontSize = Math.max(fontSize - 2, 10);
      this.textarea.style.fontSize = fontSize + "px";
    };

    // 输入区和按钮区
  const inputRow = contentEl.createDiv();
  inputRow.addClass("ask-ai-input-row");
    inputRow.style.display = "flex";
    inputRow.style.gap = "8px";
    inputRow.style.marginTop = "10px";

    // 输入框
  this.inputField = inputRow.createEl("input");
  this.inputField.type = "text";
  this.inputField.placeholder = "输入问题后按Ctrl+Enter发送";
    this.inputField.style.flex = "1";
    this.inputField.style.padding = "6px 8px";

    // 继续提问按钮
    const continueBtn = inputRow.createEl("button", { text: "提问" });
    continueBtn.style.padding = "6px 12px";
    continueBtn.style.minWidth = "60px";
    this.inputField.addEventListener("keydown", async (e) => {
      if (e.key === "Enter") {
        e.preventDefault();
        const newQuestion = this.inputField.value.trim();
        if (!newQuestion) return;
        this.messages.push({ role: "user", content: newQuestion });
        this.textarea.value += `\n\n🙋 ${newQuestion}\n\n🤖 `;
        this.inputField.value = "";
        header.setText("AI 回答中...");
        const api = this.plugin.settings.apis[
          this.apiIndex ?? this.plugin.settings.defaultApiIndex
        ];
        await this.streamApi(
          this.messages,
          api,
          this.textarea,
          header,
          newQuestion,
          this.editor,
          true
        );
        this.textarea.scrollTop = this.textarea.scrollHeight;
      }
    });

    continueBtn.onclick = async () => {
      const newQuestion = this.inputField.value.trim();
      if (!newQuestion) return;
      this.messages.push({ role: "user", content: newQuestion });
      this.textarea.value += `\n\n🙋 ${newQuestion}\n\n🤖 `;
      this.inputField.value = "";
      header.setText("AI 回答中...");
      const api = this.plugin.settings.apis[
        this.apiIndex ?? this.plugin.settings.defaultApiIndex
      ];
      await this.streamApi(
        this.messages,
        api,
        this.textarea,
        header,
        newQuestion,
        this.editor,
        true
      );
      this.textarea.scrollTop = this.textarea.scrollHeight;
    };

    // 打开时先用选中内容问一次
    try {
      const api = this.plugin.settings.apis[
        this.apiIndex ?? this.plugin.settings.defaultApiIndex
      ];
      await this.streamApi(this.messages, api, this.textarea, header, this.query, this.editor, false);
    } catch (err) {
      header.setText("❌ 请求失败");
      this.textarea.value = err.message || String(err);
    }
  }

  async streamApi(messages, api, outputEl, headerEl, selection, editor, isFollowup = false) {
    if (!api || !api.key || !api.url) {
      outputEl.value += "\n❌ API 没有配置完整";
      return;
    }

    this.abortController = new AbortController();

  let newMessages;

    if (!isFollowup) {
      // 首次提问：基于选中内容生成 prompt
      let context = "";
      if (editor) {
        const cursor = editor.getCursor();
        const startLine = Math.max(0, cursor.line - 2);
        const endLine = Math.min(editor.lineCount() - 1, cursor.line + 2);
        for (let i = startLine; i <= endLine; i++) {
          context += editor.getLine(i) + "\n";
        }
      }

  // 使用设置中的promptTemplate
  let prompt = (this.plugin.settings.promptTemplate || "").replace(/{{selection}}/g, selection).replace(/{{context}}/g, context);
  newMessages = [...messages, { role: "user", content: prompt }];
    } else {
      // 继续提问：直接用完整历史
      newMessages = [...messages];
    }

    const resp = await fetch(api.url, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        Authorization: "Bearer " + api.key,
      },
      body: JSON.stringify({
        model: api.model || "moonshot-v1-32k",
        stream: true,
        messages: newMessages,
      }),
      signal: this.abortController.signal,
    });

    if (!resp.ok || !resp.body) {
      throw new Error(`HTTP 错误: ${resp.status}`);
    }

    headerEl.setText("AI 回答：");

    const reader = resp.body.getReader();
    const decoder = new TextDecoder("utf-8");
    let buffer = "";
    let fullAnswer = "";

    while (true) {
      const { done, value } = await reader.read();
      if (done) break;
      buffer += decoder.decode(value, { stream: true });

      const parts = buffer.split("\n\n");
      buffer = parts.pop();

      for (const part of parts) {
        if (part.startsWith("data: ")) {
          const data = part.slice(6).trim();
          if (data === "[DONE]") {
            this.messages.push({ role: "assistant", content: fullAnswer });
            return;
          }
          try {
            const json = JSON.parse(data);
            const delta = json.choices?.[0]?.delta?.content;
            if (delta) {
              outputEl.value += delta;
              fullAnswer += delta;
            }
          } catch (e) {
            console.warn("解析失败:", part, e);
          }
        }
      }
    }
  }

  onClose() {
    this.contentEl.empty();
    if (this.modalEl) {
      this.modalEl.removeEventListener("mousedown", null);
    }
  }
}

class InputModal extends Modal {
  constructor(app, placeholder, onSubmit) {
    super(app);
    this.placeholder = placeholder;
    this.onSubmit = onSubmit;
  }

  onOpen() {
    const { contentEl } = this;
    contentEl.empty();

    contentEl.createEl("h3", { text: "继续提问" });

    const input = contentEl.createEl("input", {
      type: "text",
      placeholder: this.placeholder || "请输入你的问题...",
    });
    input.style.width = "100%";
    input.style.marginBottom = "10px";

    const submitBtn = contentEl.createEl("button", { text: "发送" });
    submitBtn.onclick = () => {
      const value = input.value.trim();
      if (value) {
        this.close();
        this.onSubmit(value);
      }
    };

    // 回车提交
    input.addEventListener("keydown", (e) => {
      if (e.key === "Enter") {
        submitBtn.click();
      }
    });

    input.focus();
  }

  onClose() {
    this.contentEl.empty();
  }
}


class AskAiSettingTab extends PluginSettingTab {
  constructor(app, plugin) {
    super(app, plugin);
    this.plugin = plugin;
  }

  display() {
    const { containerEl } = this;
    containerEl.empty();
    

    // 标题
    containerEl.createEl("h2", { text: "⚙️ Ask AI 设置" });
    containerEl.createEl("p", {
      text: "在这里配置多个 API，并选择一个默认 API 用于请求。",
      cls: "setting-item-description",
    });

    // Prompt模板设置
    new Setting(containerEl)
      .setName("AI Prompt模板")
      .setDesc("可自定义AI请求的prompt，支持{{selection}}和{{context}}变量。")
      .addTextArea(text => {
        text.setValue(this.plugin.settings.promptTemplate || "");
        text.inputEl.style.height = "300px";
        text.inputEl.style.width = "500px";
        text.inputEl.style.fontSize = "1em";
        text.inputEl.style.lineHeight = "1.5";
        text.onChange(async (value) => {
          this.plugin.settings.promptTemplate = value;
          await this.plugin.saveSettings();
        });
      });

    // 字体设置
    new Setting(containerEl)
      .setName("AI回答字体")
      .setDesc("输入字体名称，如 'Microsoft YaHei', 'Arial', 'SimSun' 等。留空为默认字体。")
      .addText(text => {
        text.setValue(this.plugin.settings.defaultFontFamily || "");
        text.onChange(async (value) => {
          this.plugin.settings.defaultFontFamily = value;
          await this.plugin.saveSettings();
        });
      });

    // 每个 API 用卡片包装
    this.plugin.settings.apis.forEach((api, index) => {
      const card = containerEl.createDiv("ask-ai-card");
      if (index === this.plugin.settings.defaultApiIndex) {
        card.addClass("is-default");
      }
      card.createEl("h3", {
        text: api.name || `API ${index + 1}`,
        cls: "ask-ai-card-title",
      });
      new Setting(card)
        .setName("名称")
        .addText((text) =>
          text
            .setPlaceholder("API 名称 (比如 Moonshot)")
            .setValue(api.name || "")
            .onChange(async (value) => {
              api.name = value;
              await this.plugin.saveSettings();
              this.display();
            })
        );
      new Setting(card)
        .setName("地址")
        .addText((text) =>
          text
            .setPlaceholder("API 地址 (https://...)")
            .setValue(api.url || "")
            .onChange(async (value) => {
              api.url = value;
              await this.plugin.saveSettings();
            })
        );
      new Setting(card)
        .setName("密钥")
        .addText((text) =>
          text
            .setPlaceholder("API Key (sk-xxx)")
            .setValue(api.key || "")
            .onChange(async (value) => {
              api.key = value;
              await this.plugin.saveSettings();
            })
        );
      new Setting(card)
        .setName("模型")
        .addText((text) =>
          text
            .setPlaceholder("模型 (可选)")
            .setValue(api.model || "")
            .onChange(async (value) => {
              api.model = value;
              await this.plugin.saveSettings();
            })
        );
      // 操作按钮行
      const buttonRow = card.createDiv("ask-ai-buttons");
      const defaultBtn = buttonRow.createEl("button", { text: "设为默认" });
      defaultBtn.className = "mod-cta";
      defaultBtn.onclick = async () => {
        this.plugin.settings.defaultApiIndex = index;
        await this.plugin.saveSettings();
        new Notice(`已将 ${api.name || "API " + (index + 1)} 设为默认`);
        this.display();
      };
      const deleteBtn = buttonRow.createEl("button", { text: "删除" });
      deleteBtn.className = "mod-warning";
      deleteBtn.onclick = async () => {
        this.plugin.settings.apis.splice(index, 1);
        await this.plugin.saveSettings();
        new Notice(`已删除 ${api.name || "API " + (index + 1)}`);
        this.display();
        if (this.plugin.settings.defaultApiIndex >= this.plugin.settings.apis.length) {
          this.plugin.settings.defaultApiIndex = 0;
        }
        await this.plugin.saveSettings();
        this.display();
      };
    });

    // 添加按钮
    const addRow = containerEl.createDiv("ask-ai-add-row");
    const addBtn = addRow.createEl("button", { text: "➕ 添加 API" });
    addBtn.className = "mod-cta";
    addBtn.onclick = async () => {
      this.plugin.settings.apis.push({
        name: "New API",
        url: "",
        key: "",
        model: ""
      });
      await this.plugin.saveSettings();
      this.display();
    };
  }
}